/*
 * If distributed as part of the Linux kernel, this code is licensed under the
 * terms of the GPL v2.
 *
 * Otherwise, the following license terms apply:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1) Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3) The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESSED OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <linux/types.h>
#include <stdarg.h>
#include <linux/linkage.h>
#include <linux/printk.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <asm/io.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/err.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <linux/module.h>
//#include <asm/hw_irq.h>
#include <linux/irq.h>
#include <linux/irqdomain.h>
#include "fcuvar.h"
#include "fcureg.h"
#include "fcu_subr.h"			/* for plist_t */
#include "fcucom.h"
#include "fax_share.h"		/* XXX */

#define DRIVER_NAME	"fcu"		/* ドライバ名 */

extern struct devlist fax_devlist[2];
extern ulong FBI_COMMON_AREA_BASE;


/*
 * plist_t count
 */
#define PLIST_CNT	8			/* packet FIFOの個数 */

/*
 * packet size buffer
 */
typedef struct pkt_buf {
	u_char	buf[FCUCMDSIZE];
} pkt_buf_t;


struct fcu_softc {
	fax_driver_header_t		sc_header;
	struct pci_dev                  *pci_dev;
	u_short			sc_flags;	/* flags */
#define FSF_ISOPEN		0x0001	/* open されている */
#define FSF_RSLEEP		0x0002	/* 受信データ待ち sleep 中 */
#define FSF_WSLEEP		0x0004	/* 送信バッファ待ち sleep 中 */
#define FSF_RSELWANT	0x0008	/* 受信データselect要求 */
#define FSF_WSELWANT	0x0010	/* 送信fifo空きselect要求 */
#define FSF_WRITING		0x0020	/* 送信データ書き込み中 */
#define FSF_ACKREQ		0x0040	/* ACK保留中 */
#define FSF_INITIAL		0x0080	/* 初期化処理中 */
#define FSF_FCURDY		0x0100	/* FCU READY確認割込み起動済 */
#define FSF_RDYINTR		0x0200	/* READY確認割込み受信済 */
#define FSF_REBOOT		0x0400	/* FCUリブート検出 */
	plist_t			sc_rpl;		/* read  FIFO */
	plist_t			sc_wpl;		/* write FIFO */
	pkt_buf_t		sc_rbuf;	/* read  work buffer */
	pkt_buf_t		sc_wbuf;	/* write work buffer */
	pkt_buf_t		sc_irbuf;	/* intr read  work buffer */
	pkt_buf_t		sc_iwbuf;	/* intr write work buffer */
	fax_share_param_t	*sc_share;
	unsigned long		io_vaddr;
	unsigned long		io_addr;
	unsigned long		io_len;
	unsigned long		mem_vaddr;
	unsigned long		mem_addr;
	unsigned long		mem_len;
	unsigned int		irqno;
};

typedef struct fcu_softc	fcu_t;

fcu_t g_fcu_softc;
EXPORT_SYMBOL(g_fcu_softc);
static fcu_t *sc = &g_fcu_softc;

#ifdef FCU_STR_SUPPORT
#ifdef USE_PCI_CONF_RECOVER
/*
 * コンフィグ保存領域
 */
static pciconf_t fcu_pci_conf;
#endif	/* !USE_PCI_CONF_RECOVER */

/*
 * 省エネ対応したFCU(FACE3.5)装着有無情報
 * 1:あり
 * 0:なし
 */
static int is_fcu_face35 = 0;
#endif	/* !FCU_STR_SUPPORT */

#ifdef FCU_STR_SUPPORT
int	fcu_chaeck_face35(void);
void	fcu_pci_conf_recover(void);
#endif	/* !FCU_STR_SUPPORT */

/*
 * prototypes
 */
static int fcuopen( struct inode *inode, struct file *filp );
static int fcuclose( struct inode *inode, struct file *filp );
static ssize_t fcuread( struct file* p_file, char* buf, size_t count, loff_t* f_pos );
static ssize_t fcuwrite( struct file* p_file, const char* buf, size_t count, loff_t* f_pos );
static long fcuioctl( struct file *filp, unsigned int cmd, unsigned long arg);
/*
 * static prototypes
 */

struct file_operations fcu_FileOps = {
 .owner			= THIS_MODULE,
 .open			= fcuopen,
 .release		= fcuclose,
 .read			= fcuread,
 .write			= fcuwrite,
 .unlocked_ioctl= fcuioctl,
};
EXPORT_SYMBOL(fcu_FileOps);

#ifdef	FCU_DEBUG_IOCTL
static void fcu_debug_0(fcu_t *);
static void fcu_debug_ioctl_func(fcu_t *, int);
#endif	/* !FCU_DEBUG_IOCTL */

static int fcu_int_set(u_char);
static u_char fcu_int_conf(void);
static fcu_mirror_conf_t fcu_mirror_conf(void);
static void fcu_mirror_rsp(fcu_mirror_rsp_t);

/*
extern int fcudstart(struct pci_dev*, void*, void*, unsigned int);
extern irqreturn_t fcudintr(void*);
*/
static __inline void fcu_fgate_assert_bugfix(fcu_t *sc);

/********************************************************************/
/* PCI BASE領域全クリア初期化用 */
static __inline void
init_pcibase_intr(fcu_t *sc)
{
	unsigned long ofs;
	for(ofs = 0x00000000; ofs < FBI_OFS_DP_BASE_SIZE; ofs += 4) {
		bsw4_pcibase_intrclear(sc, FBI_OFS_DP_INTR_TOP + ofs);
	}
}

static __inline void
fcu_write_dpram(fcu_t *sc, u_char *ptr)
{
	int i;
	u_long *p = (u_long *)ptr;		/* big endian */
	u_long *be_chg = (u_long *)ptr;

	/*
	 * DPRAMのバイトレーン制御対応
	 */
	for (i = 0; i < FCUCMDSIZE; i += 4, be_chg++) {
		*be_chg = cpu_to_be32(*be_chg);
	}

	/* キャッシュフラッシュの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */

	for (i = 0; i < FCUCMDSIZE; i += 4, p++) {
		bsw4_pcibase(sc, FBI_OFS_DP_PACKET + i, *p);
#if 1 /* for debug */
		PRINT_DEBUG("write dpram value=%08lx\n", *p);
#endif
	}
#if 1 /* for debug */
		PRINT_DEBUG("\n");
#endif
	/* ライト(パケットSCU)をFCUに通知 */
	fcu_int_set(FCUD_INTR_WRITE_PACKET_SCU);
}

static __inline void
fcu_read_dpram(fcu_t *sc, u_char *ptr)
{
	int i;
	u_long *p = (u_long *)ptr;		/* big endian */

	u_long *be_chg = (u_long *)ptr;

	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */

	for (i = 0; i < FCUCMDSIZE; i += 4, p++){
		*p = bsr4_fcubase(sc, FBI_OFS_DP_PACKET + i);
#if 0 /* for debug */
		PRINT_DEBUG("read dpram value=%08lx\n", *p);
#endif		
	}
	bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_ACK);

	/*
	 * DPRAMのバイトレーン制御対応
	 */
	for (i = 0; i < FCUCMDSIZE; i += 4, be_chg++) {
		*be_chg = cpu_to_be32(*be_chg);
	}

}

/********************************************************************/

/**
 * FCUドライバの管理情報を初期化する関数
 */
static void fcu_initial(void){

	/**
	 * 全体をゼロクリア
	 */
	memset(sc, 0, sizeof(fcu_t));

	/**
	 * ゼロクリア以外に必要な初期化
	 */
	FLAG_SET(FSF_INITIAL);

}

/********************************************************************/

/*
 * attach
 */
int fcu_attach(void *self){

	int error = -1;
	
	PRINT_INFO("fcu_attach: fcu initialization is started...\n");
	
	// init parameter 
	fcu_initial();
	
	// get shared parameter
	sc->sc_share = get_fax_share_param();
	
	// enable pci … 削除

	// enable bus_master … 削除

	// io mapping … 削除
	
	// memory mapping
	sc->mem_addr = FBI_COMMON_AREA_BASE;
	sc->mem_len = FBI_OFS_DP_BASE_SIZE * 2;
	
	PRINT_INFO("mem_addr=%lx, mem_len=%ld\n", sc->mem_addr, sc->mem_len);

	if(!request_mem_region(sc->mem_addr, sc->mem_len, DEV_NAME)){
		PRINT_ERR("%s: request_region_mem error =%d\n", DRIVER_NAME, error);
		error = -1;
		goto mem_request_error;
	}
	
	sc->mem_vaddr = (unsigned long)ioremap_nocache(sc->mem_addr, sc->mem_len);
	if(!sc->mem_vaddr){
		PRINT_ERR("%s: remap_mem error\n", DRIVER_NAME);
		error = -1;
		goto mem_remap_error;
	}
	PRINT_INFO("mem_vaddr=%lx\n", sc->mem_vaddr);
	
	/*
	 * DPRAM割り込み要因エリアをクリア
	 */
	init_pcibase_intr(sc);
	
	/*
	 * DMA割り込みマスクとバイトレーンをリセット … 削除
	 */

	/*
	 * DMAを停止 … 削除
	 */
	
	/*
	 * 割り込みハンドラを登録 … 削除
	 */
	
	/*
	 * fifo allocate … 削除
	 */
		
	// キャラクタデバイス系の処理は共通部に移管
	
	// 共通部に値をセット
//	sc->sc_share->io_vaddr = sc->io_vaddr;
//	sc->sc_share->io_addr = sc->io_addr;
//	sc->sc_share->io_len = sc->io_len;
	sc->sc_share->mem_vaddr = sc->mem_vaddr;
	sc->sc_share->mem_addr = sc->mem_addr;
	sc->sc_share->mem_len = sc->mem_len;
//	sc->sc_share->irqno = sc->irqno;
//	sc->sc_share->fax_intr_establish = fcud_intr_establish;
	
	PRINT_INFO("fcu attached\n");

	return 0;
	
//fifo_allocate_error:
//	free_irq(sc->irqno, NULL);
//request_irq_error:
//	iounmap((void *)sc->mem_vaddr);
mem_remap_error:
	release_mem_region(sc->mem_addr, sc->mem_len);
mem_request_error:
// iounmap((void *)sc->io_vaddr);
// io_remap_error:
// release_mem_region(sc->io_addr, sc->io_len);
// io_request_error:
// pci_disable_device(sc->pci_dev);
// pci_enable_error:
	return error;
	
}
EXPORT_SYMBOL(fcu_attach);

/*
 * detach
 */
int fcu_detach(void *self){

	PRINT_INFO("%s: fcu detach\n", DRIVER_NAME);
	
//	free_irq(sc->irqno, NULL);
	iounmap((void *)sc->mem_vaddr);
	release_mem_region(sc->mem_addr, sc->mem_len);
//	iounmap((void *)sc->io_vaddr);
//	release_mem_region(sc->io_addr, sc->io_len);
//	pci_disable_device(sc->pci_dev);
	memset(sc, 0, sizeof(fcu_t));
	return 0;
}
EXPORT_SYMBOL(fcu_detach);

/*
 * open
 */
static int fcuopen( struct inode* inode, struct file* p_file )
{

	PRINT_INFO("%s: [fcuopen start]\n", DRIVER_NAME);
	PRINT_DEBUG("%s: [fcuopen end]\n", DRIVER_NAME);

	return (0);
}
EXPORT_SYMBOL(fcuopen);


/*
 * close
 */
static int fcuclose( struct inode* inode, struct file* p_file )
{
	PRINT_INFO("%s: [fcuclose start]\n", DRIVER_NAME);


	/*
	 * レディビットをリセット … 削除
	 */

	/*
	 * plistの初期化 … 削除
	 */

	/*
	 * sc_flagsのリセット
	 */
	sc->sc_flags = FSF_INITIAL;

	PRINT_DEBUG("%s: [fcuclose end]\n", DRIVER_NAME);

	return (0);
}
EXPORT_SYMBOL(fcuclose);


/*
 * write
 */
static ssize_t fcuwrite( struct file* p_file, const char* buf, size_t count, loff_t* f_pos )
{
	int error;

	PRINT_DEBUG("%s: device write\n", DRIVER_NAME);

	if (count < FCUCMDSIZE){
		PRINT_ERR("command size error %d\n", count);
		return (-EINVAL);
	}

	error = copy_from_user((caddr_t)sc->sc_wbuf.buf, buf, FCUCMDSIZE);
	if (error) {
		PRINT_ERR("copy_from_user error\n");
		return (error);
	}
	
	fcu_write_dpram(sc, sc->sc_wbuf.buf);

	return (0);
}
EXPORT_SYMBOL(fcuwrite);


/*
 * read
 */
static ssize_t fcuread( struct file* p_file, char* buf, size_t count, loff_t* f_pos )
{
	int error;

	PRINT_DEBUG("%s: device read\n", DRIVER_NAME);

	if (count < FCUCMDSIZE) {
		PRINT_ERR("command size error%d\n", count);
		return (-EINVAL);
	}
	
	fcu_read_dpram(sc, sc->sc_rbuf.buf);

	error = copy_to_user(buf, (caddr_t)sc->sc_rbuf.buf, FCUCMDSIZE);
	if( error ){
		PRINT_ERR("copy_to_user error\n");
		return (-EFAULT);
	}

	return (FCUCMDSIZE);
}
EXPORT_SYMBOL(fcuread);

/*
 * FBI不具合対策
 * スキャン開始時FGATEアサート状態を回避する処理
 */
static __inline void
fcu_fgate_assert_bugfix(fcu_t *sc)
{
	/*
	 * (1)DMA1 FIFOをダミーライト
	 * (2)スキャナデータ転送終了コマンド1回目
	 * (3)スキャナデータ転送終了コマンド2回目
	 * (念のために割り込み禁止にして処理する)
	 */
	lpux_bsw32_mem(0, (void *)(sc->mem_vaddr + FBI_OFS_MM_DMA1_BASE));
	bsw4_io(sc, FBI_OFS_IO_SCANENDCMD, 0);
	bsw4_io(sc, FBI_OFS_IO_SCANENDCMD, 0);
}

/*
 * ready check
 */
static __inline u_long
fcu_get_ready(fcu_t *sc)
{
	u_long ret = FCU_NOTREADY;
	u_long rdybit;

	rdybit = bsr4_io(sc, FBI_OFS_IO_RESETCTL);/* FBI bug : half => long */
	PRINT_INFO("fcu_get_ready: reset control register %lx.\n", rdybit);

	/*
	 * 自分のレディをセットしていない状態
	 */
	if (!FBI_IS_PCIREADY(rdybit))
		return (ret);


	/*
	 * FCUレディを確認していない
	 */
	if (!FLAG_CHK(FSF_FCURDY) && FBI_IS_FCUREADY(rdybit)) {
		FLAG_SET(FSF_FCURDY);		/* FCUレディを確認した */

		/*
		 * FBI不具合対策
		 */
		fcu_fgate_assert_bugfix(sc);

		/*
		 * FCUレディ割込みをセット
		 */
		bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_READY);
		/*
		 * DPRAM割込み起動
		 */
		bsw4_io(sc, FBI_OFS_IO_INTRCTL, FBI_BIT_INTRCTL_INTR);
	}

	/*
	 * FCUレディ割込み起動後 && SCUレディ確認割込み受信後
	 */
	if (FLAG_CHK(FSF_FCURDY|FSF_RDYINTR) == (FSF_FCURDY|FSF_RDYINTR))
		ret = FCU_READY;

	/*
	 * 両方の割込みが揃った
	 */
	if (ret == FCU_READY) {
		FLAG_CLR(FSF_INITIAL);		/* 初期化中状態終わり */
		FLAG_CLR(FSF_REBOOT);		/* FCUリブート検出終わり */
	}


	return (ret);
}

/*
 * interrupt factor set
 * 戻り値
 * 0：正常
 * 0以外：異常
 */
static int fcu_int_set(u_char intSetParam){
	int ret = 0;
	switch (intSetParam) {
	case FCUD_INTR_WRITE_PACKET_SCU:
		/*
		 * シーケンス崩れチェック。
		 * CTLからWRITEをセットする時点で, FCUからのACKが設定されていたら
		 * 明らかに異常な状態
		 */
		if(bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK)){
			PRINT_EMERG("!!!!!ack already set when write send!!!!!\n");
		}
		bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_WRITE);
		/* キャッシュフラッシュの実施は不要 */
		/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
		break;
	case FCUD_INTR_ACK_PACKET_FCU:
		bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_ACK);
		/* キャッシュフラッシュの実施は不要 */
		/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
		break;
	case FCUD_INTR_FCU_RDY_CONF:
		bsw1_pcibase_intr(sc, FBI_OFS_DP_INTR_READY);
		/* キャッシュフラッシュの実施は不要 */
		/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
		break;
	default:
		// ありえない
		PRINT_ERR("fcu_int_set irregal value error = %d\n", intSetParam);
		ret = (-EINVAL);
		break;
	}
	return ret;
}

/*
 * interrupt factor conf
 */
static u_char fcu_int_conf(void){
	
	u_char factor = 0;
	
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */

	if(bsr1_fcubase(sc, FBI_OFS_DP_INTR_WRITE)){
		bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_WRITE);
		factor = FCUD_INTR_WRITE_PACKET_FCU;
		PRINT_DEBUG("fcu_int_conf write_packet_fcu\n");
	} else if(bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK)){
		bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_ACK);
		factor = FCUD_INTR_ACK_PACKET_SCU;
		PRINT_DEBUG("fcu_int_conf ack_packet_fcu\n");
	} else if(bsr1_fcubase(sc, FBI_OFS_DP_INTR_READY)){
		bsw1_fcubase_intrclear(sc, FBI_OFS_DP_INTR_READY);
		factor = FCUD_INTR_SCU_RDY_CONF;
		PRINT_DEBUG("fcu_int_conf scu_ready_conf\n");
	} else {
		// ありえない
		PRINT_ERR("fcu_int_conf no_int_factor\n");
	}
	
	return factor;
}

/*
 * mirror conf
 */
static fcu_mirror_conf_t fcu_mirror_conf(void){
	
	/* キャッシュインバリデートの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */

	fcu_mirror_conf_t ret;
	
	ret.device = bsr1_fcubase(sc, FBI_OFS_DP_MIRROR_DEVICE);
	ret.direction = bsr1_fcubase(sc, FBI_OFS_DP_MIRROR_DIRECTION);
	ret.ddr_adr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_MIRROR_DDRADR));
	ret.block_adr = cpu_to_be32(bsr4_fcubase(sc, FBI_OFS_DP_MIRROR_BLOCKADR));
	ret.size = cpu_to_be16(bsr2_fcubase(sc, FBI_OFS_DP_MIRROR_SIZE));
	
	PRINT_DEBUG("fcu_mirror_conf device = %d\n", ret.device);
	PRINT_DEBUG("fcu_mirror_conf direction = %d\n", ret.direction);
	PRINT_DEBUG("fcu_mirror_conf ddr_adr = %08lx\n", ret.ddr_adr);
	PRINT_DEBUG("fcu_mirror_conf block_adr = %08lx\n", ret.block_adr);
	PRINT_DEBUG("fcu_mirror_conf size = %04x\n", ret.size);
	
	return ret;
}

/*
 * mirror rsp
 */
static void fcu_mirror_rsp(fcu_mirror_rsp_t mirrorRespParam){
		
	bsw1_pcibase(sc, FBI_OFS_DP_MIRROR_RESULT, mirrorRespParam.result);
	/* キャッシュフラッシュの実施は不要 */
	/* 共有メモリ空間はioremap_nocache()によりキャッシュ不可としているため */
}

/*
 * ioctl
 */
static long fcuioctl( struct file* p_file, unsigned int cmd, unsigned long arg )
{

	
	u_long fcuready;
	u_char intSetParam;
	u_char intConfParam;
	fcu_mirror_conf_t mirrorConfParam;
	fcu_mirror_rsp_t mirrorRespParam;
	int fcuActive;
	
	int error = 0;
	PRINT_DEBUG( "%s: fcuioctl\n", DRIVER_NAME);

	switch (cmd) {
	case FIOCSETRDY:
		/* FBI bug : half => long */
		PRINT_INFO("[fcuioctl][%s] FIOCSETRDY\n", DRIVER_NAME);
		bsw4_io(sc, FBI_OFS_IO_RESETCTL, FBI_BIT_RESETCTL_PCIRDY);
		break;
	case FIOCGETRDY:
		PRINT_INFO("[fcuioctl][%s] FIOCGETRDY\n", DRIVER_NAME);
		fcuready = fcu_get_ready(sc);
		PRINT_INFO("fcu_get_ready val=%ld\n", fcuready);
		error = copy_to_user((u_long *)arg, &fcuready, sizeof(u_long));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;
	case FIOCDEBUG:
		PRINT_DEBUG("[fcuioctl][%s] FIOCDEBUG\n", DRIVER_NAME);
#ifdef	FCU_DEBUG_IOCTL
		fcu_debug_ioctl_func(sc, *(int *)arg);
#endif	/* !FCU_DEBUG_IOCTL */
		break;
	case FIOCINTSET:
		PRINT_DEBUG("[fcuioctl][%s] FIOCINTSET\n", DRIVER_NAME);
		error = copy_from_user(&intSetParam, (void *)arg, sizeof(u_char));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		error = fcu_int_set(intSetParam);
		if ( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid int param=%d\n", DRIVER_NAME, error);
			return (error);
		}	
		break;
	case FIOCINTCONF:
		PRINT_DEBUG("[fcuioctl][%s] FIOCINTCONF\n", DRIVER_NAME);
		intConfParam = fcu_int_conf();
		error = copy_to_user((void *)arg, &intConfParam, sizeof(u_char));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;
	case FIOCMIRRORCONF:
		PRINT_DEBUG("[fcuioctl][%s] FIOCMIRRORCONF\n", DRIVER_NAME);
		mirrorConfParam = fcu_mirror_conf();
		error = copy_to_user((void *)arg, &mirrorConfParam, sizeof(fcu_mirror_conf_t));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;
	case FIOCMIRRORRESP:
		PRINT_DEBUG("[fcuioctl][%s] FIOCMIRRORRESP\n", DRIVER_NAME);
		error = copy_from_user(&mirrorRespParam, (void *)arg, sizeof(fcu_mirror_rsp_t));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		fcu_mirror_rsp(mirrorRespParam);
		break;
	case FIOCFCUACTIVE:
		PRINT_INFO("[fcuioctl][%s] FIOCFCUACTIVE\n", DRIVER_NAME);
		fcuActive = isFcuActive();
		error = copy_to_user((void *)arg, &fcuActive, sizeof(int));
		if( error ){
			PRINT_ERR("[fcuioctl][%s] Invalid argument\n", DRIVER_NAME);
			return (error);
		}
		break;
	default:
		PRINT_ERR("[fcuioctl][%s] UNKNOWN\n", DRIVER_NAME);
		return (-EINVAL);
	}

	return (error);
}
EXPORT_SYMBOL(fcuioctl);


int fcu_pci_runtime_suspend(struct device* dev){
	PRINT_DEBUG("fcu_pci_runtime_suspend called\n");
	FLAG_CLR(FSF_FCURDY);
	return 0;
}
EXPORT_SYMBOL(fcu_pci_runtime_suspend);

int fcu_pci_runtime_resume(struct device* dev){
	PRINT_DEBUG("fcu_pci_runtime_resume calledd\n");
	return 0;
}
EXPORT_SYMBOL(fcu_pci_runtime_resume);

/********************************************************************/
#ifdef	FCU_DEBUG_IOCTL
static void
fcu_debug_0(fcu_t *sc)
{
	fbi_intr_t ic;
	u_long rdybit;

	rdybit = bsr4_io(sc, FBI_OFS_IO_RESETCTL);
	printk("rdy=%x\n", (int)rdybit);


	printk("\n[$Id: fcu.c,v 1.11 2011/12/08 00:31:26 takubo Exp $]\n");

	printk("[softc]\n");

	printk("[sc_flags]\n");
	printk("%s", FLAG_CHK(FSF_ISOPEN)   ? "FSF_ISOPEN\n"   : "");
	printk("%s", FLAG_CHK(FSF_RSLEEP)   ? "FSF_RSLEEP\n"   : "");
	printk("%s", FLAG_CHK(FSF_WSLEEP)   ? "FSF_WSLEEP\n"   : "");
	printk("%s", FLAG_CHK(FSF_RSELWANT) ? "FSF_RSELWANT\n" : "");
	printk("%s", FLAG_CHK(FSF_WSELWANT) ? "FSF_WSELWANT\n" : "");
	printk("%s", FLAG_CHK(FSF_WRITING)  ? "FSF_WRITING\n"  : "");
	printk("%s", FLAG_CHK(FSF_ACKREQ)   ? "FSF_ACKREQ\n"   : "");
	printk("%s", FLAG_CHK(FSF_INITIAL)  ? "FSF_INITIAL\n"  : "");
	printk("%s", FLAG_CHK(FSF_FCURDY)   ? "FSF_FCURDY\n"   : "");
	printk("%s", FLAG_CHK(FSF_RDYINTR)  ? "FSF_RDYINTR\n"  : "");
	printk("%s", FLAG_CHK(FSF_REBOOT)   ? "FSF_REBOOT\n"   : "");

	printk("[plist]\n");
	printk("plist r cnt : %d\n", plcnt(&sc->sc_rpl));
	printk("plist w emp : %d\n", plemp(&sc->sc_wpl));

	ic.intr = bsr4_io(sc, FBI_OFS_IO_INTRCAUSE);
	ic.cmd  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_WRITE);
	ic.ack  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_ACK);
	ic.rdy  = bsr1_fcubase(sc, FBI_OFS_DP_INTR_READY);
	ic.dma3 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA3);
	ic.dma4 = bsr1_fcubase(sc, FBI_OFS_DP_INTR_DMA4);

	printk("[intr cause & DPRAM]\n");
	printk("ic.intr=%x\n", (int)ic.intr);
	printk("    cmd=%x\n", (int)ic.cmd);
	printk("    ack=%x\n", (int)ic.ack);
	printk("    rdy=%x\n", (int)ic.rdy);
	printk("   dma3=%x\n", (int)ic.dma3);
	printk("   dma4=%x\n", (int)ic.dma4);

	printk("[bsr4_io(sc,FBI_OFS_IO_DMACTL)]\n");
	printk("%08x \n",bsr4_io(sc,FBI_OFS_IO_DMACTL));

}


static void fcu_debug_ioctl_func(fcu_t *sc, int arg)
{
	switch (arg) {
	default:
	case 0:
		fcu_debug_0(sc);
		break;
	}
}

#endif	/* !FCU_DEBUG_IOCTL */

